/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

static const char __idstring[] = "@(#)$Id: mx_ether_common.c,v 1.14.6.3 2006/12/18 20:22:20 loic Exp $";

#include "mx_arch.h"
#include "mx_misc.h"
#include "mx_instance.h"
#include "mx_malloc.h"
#include "mx_ether_common.h"
#include "mx_pio.h"
#include "mx_peer.h"


static void
mx_ethernet_free_descriptors(mx_instance_state_t *is)
{
	struct mx_ether *ether = is->ether;
	
	
	if (ether->rx_small.shadow) {
		mx_kfree(ether->rx_small.shadow);
		ether->rx_small.shadow = 0;
	}
	if (ether->rx_big.shadow) {
		mx_kfree(ether->rx_big.shadow);
		ether->rx_big.shadow = 0;
	}
	is->ether_is_open = 0;
}

static int
mx_ethernet_alloc_descriptors(mx_instance_state_t *is)
{
	struct mx_ether *ether = is->ether;
	
	
	if (ether->rx_small.shadow || ether->rx_big.shadow) {
		return EBUSY;
	}
	
	ether->rx_small.shadow = mx_kmalloc(MX_VPAGE_SIZE, MX_MZERO|MX_WAITOK);
	ether->rx_big.shadow = mx_kmalloc(MX_VPAGE_SIZE, MX_MZERO|MX_WAITOK);
	
	if (!ether->rx_small.shadow || !ether->rx_big.shadow) {
		MX_WARN (("mx%d: unable to allocate shadow descriptor rings\n",
			  is->id));
		mx_ethernet_free_descriptors(is);
		return ENOMEM;
	}
	return 0;
}


/* it is expected that a driver would call this before
   allocating its receive buffers */

int
mx_ether_open_common(mx_instance_state_t *is, int mtu, int max_small_rx, 
		     int max_big_rx)
{
	struct mx_ether *ether = is->ether;
	uint32_t ethernet_tx_ring, ethernet_tx_cnt;
	uint32_t ethernet_rx_small_ring, ethernet_rx_small_cnt;
	uint32_t ethernet_rx_big_ring, ethernet_rx_big_cnt;
	int status;

	status = 0;

	/* set the static parameters */
	status |= mx_mcpi.set_param(is->id, is->lanai.sram, 
				    "ethernet_bigbuf", max_big_rx);
	if (status) {
		MX_WARN(("mx%d: Could not set ethernet related params\n",
			 is->id));
		goto abort_with_nothing;
	}
	status |= mx_mcpi.get_param(is->id, is->lanai.sram, 
				    "ethernet_tx_ring",
				    &ethernet_tx_ring);

	status |= mx_mcpi.get_param(is->id, is->lanai.sram, 
				    "ethernet_tx_cnt",
				    &ethernet_tx_cnt);

	status |= mx_mcpi.get_param(is->id, is->lanai.sram, 
				    "ethernet_rx_small_ring",
				    &ethernet_rx_small_ring);

	status |= mx_mcpi.get_param(is->id, is->lanai.sram, 
				    "ethernet_rx_small_cnt",
				    &ethernet_rx_small_cnt);

	status |= mx_mcpi.get_param(is->id, is->lanai.sram, 
				    "ethernet_rx_big_ring",
				    &ethernet_rx_big_ring);

	status |= mx_mcpi.get_param(is->id, is->lanai.sram, 
				    "ethernet_rx_big_cnt",
				    &ethernet_rx_big_cnt);

	if (status) {
		MX_WARN(("mx%d: Could not get ethernet related pointers\n",
			 is->id));
		goto abort_with_nothing;
	}
	ether->tx.ring = (mcp_kreq_ether_send_t *)
		((char *)is->lanai.sram + ethernet_tx_ring);
	ether->tx.lanai_cnt = (uint32_t *)
		((char *)is->lanai.sram + ethernet_tx_cnt);
	ether->rx_small.ring = (mcp_kreq_ether_recv_t *)
		((char *)is->lanai.sram + ethernet_rx_small_ring);

	ether->rx_small.lanai_cnt = (uint32_t *)
		((char *)is->lanai.sram + ethernet_rx_small_cnt);

	ether->rx_big.ring = (mcp_kreq_ether_recv_t *)
		((char *)is->lanai.sram + ethernet_rx_big_ring);
	ether->rx_big.lanai_cnt = (uint32_t *)
		((char *)is->lanai.sram + ethernet_rx_big_cnt);

	/* clear the mcp's polling variables */
	MX_PIO_WRITE(ether->tx.lanai_cnt, 0);
	MX_PIO_WRITE(ether->rx_small.lanai_cnt, 0);
	MX_PIO_WRITE(ether->rx_big.lanai_cnt, 0);

	status = mx_ethernet_alloc_descriptors(is);
	if (status) {
		MX_WARN(("mx%d: alloc descriptors failed, errno = %d\n",
			 is->id, status));
		goto abort_with_nothing;
	}

	mx_sync_init(&ether->cmd_sync, is, -1, "ether cmd sync");

	/* reset indexes */
	ether->tx.req = 0;
	ether->tx.done = 0;
	ether->rx_big.cnt = 0;
	ether->rx_small.cnt = 0;
	is->ether_is_open = 1;

	return 0;

abort_with_nothing:
	return status;
}

void
mx_ether_start_common(mx_instance_state_t *is, int mtu, int max_small_rx, 
	       int max_big_rx)
{
	int status = 0;
	status |= mx_mcpi.set_param(is->id, is->lanai.sram, 
				    "ethernet_mtu", mtu);
	status |= mx_mcpi.set_param(is->id, is->lanai.sram, 
				    "ethernet_smallbuf", max_small_rx);
	if (status) {
		MX_WARN(("Cannot set mtu/ethernet_smallbuf:something seriously broken\n"));
	}
}

/* it is expected that a driver would call this after disabling
   ethernet and freeing its recv buffers */
void
mx_ether_close_common(mx_instance_state_t *is)
{
	struct mx_ether *ether = is->ether;


	mx_sync_destroy(&ether->cmd_sync);
	mx_ethernet_free_descriptors(is);
}

void 
mx_ether_set_mac_address_common(struct mx_ether *sc, uint8_t *addr)
{
	static int warned = 0;

	if (!warned) {
		MX_WARN(("Setting ether mac address not yet supported\n"));
		warned++;
	}
}

void 
mx_ether_set_promisc_common(struct mx_ether *sc, int promisc)
{
	static int warned = 0;

	if (!warned && promisc) {
		MX_WARN(("Setting/clearing promiscuous not yet supported\n"));
		MX_WARN(("Device is currently always promiscuous\n"));
		warned++;
	}
}

void mx_ether_gw_rx(mx_instance_state_t *is, void *pkt)
{
	struct {
		uint16_t gw_high16;
		uint16_t gw_middle16;
		uint16_t gw_low16;
		uint8_t dst[6];
		uint16_t src_high16;
		uint16_t src_middle16;
		uint16_t src_low16;
	} *h = pkt;
	mx_peer_hash_t *gw, *src;
	uint16_t gw_high16, src_high16;
	uint32_t gw_low32, src_low32;

	gw_high16 = ntohs(h->gw_high16) ;
	gw_low32 = (ntohs(h->gw_middle16)  << 16) + ntohs(h->gw_low16);
	src_high16 = ntohs(h->src_high16) ;
	src_low32 = (ntohs(h->src_middle16)  << 16) + ntohs(h->src_low16);

	gw = mx_peer_lookup(gw_high16, gw_low32);
	if (!gw) {
		static int done;
		if (done++ < 10) {
			MX_WARN(("bad gw:%02x%04x src=%02x%04x\n", gw_high16, gw_low32,
				 src_high16, src_low32));
		}
		return;
	}

	src = mx_peer_lookup_eth(src_high16, src_low32, 1);
	if (src)
		src->gw = gw->index;
}

/*
  This file uses MX driver indentation.

  Local Variables:
  c-file-style:"linux"
  tab-width:8
  End:
*/
